<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Catálogo Interactivo - Tu Negocio</title>
    <!-- Carga de Tailwind CSS para estilos de utilidad -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Carga de Font Awesome para iconos -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <!-- Preconexión para fuentes de Google Fonts -->
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <!-- Carga de la fuente 'Barlow' desde Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Barlow:wght@400;700;900&display=swap" rel="stylesheet">
    <style>
        /* Definición de variables CSS personalizadas para los colores de la marca */
        :root {
            --brand-dark-blue: #07162d; /* Primario: Inteligencia */
            --brand-yellow: #ffc857;    /* Resalte: Visión */
            --brand-red-accent: #9e1428; /* Secundario: Creatividad */
            --brand-light-gray: #f5f5f5; /* Fondo: Honestidad */
            --brand-darkest-blue: #040d1a;/* Negro para fondos */
            --brand-dark-text: var(--brand-dark-blue); /* Color de texto principal */
        }

        /* Estilos globales aplicados al cuerpo del documento */
        body {
            font-family: 'Barlow', sans-serif; /* Fuente principal */
            background-color: var(--brand-light-gray); /* Color de fondo */
            color: var(--brand-dark-text); /* Color de texto */
            line-height: 1.6; /* Altura de línea para mejor legibilidad */
        }
        
        /* Estilos para títulos y elementos con la clase 'font-heading' */
        h1, h2, h3, .font-heading {
            font-family: 'Barlow', sans-serif;
            font-weight: 900 !important; /* Negrita extra */
            text-transform: uppercase; /* Texto en mayúsculas */
            letter-spacing: 1px; /* Espaciado entre letras */
        }

        /* Animación de desvanecimiento para elementos que aparecen */
        .fade-in { animation: fadeIn 0.5s ease-in-out; }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        /* Estilos específicos para las tarjetas de producto */
        .product-card {
            background: white;
            border-radius: 0.75rem; /* Bordes redondeados */
            text-align: center;
            box-shadow: 0 15px 25px -5px rgba(0, 0, 0, 0.15), 0 8px 10px -4px rgba(0, 0, 0, 0.08); /* Sombra suave */
            transition: transform 0.3s ease, box-shadow 0.3s ease; /* Transición para hover */
            height: 100%; /* Asegura que las tarjetas tengan la misma altura en la cuadrícula */
            display: flex;
            flex-direction: column; /* Organiza el contenido en columna */
            overflow: hidden; /* Oculta contenido que se desborda */
        }
        .product-card:hover {
            transform: translateY(-12px); /* Efecto de elevación al pasar el mouse */
            box-shadow: 0 30px 45px -8px rgba(0, 0, 0, 0.2), 0 15px 20px -6px rgba(0, 0, 0, 0.1); /* Sombra más pronunciada */
        }
        .product-card .title {
            font-size: 1.5rem;
            color: var(--brand-dark-blue);
            font-weight: 900 !important;
        }
        
        /* Estilos base para botones de acción */
        .btn-action {
            padding: 0.75rem 1.5rem;
            border-radius: 0.5rem;
            font-weight: 700;
            transition: all 0.3s ease;
            text-transform: uppercase;
            box-shadow: 0 4px 8px rgba(0,0,0,0.15);
            display: inline-flex; /* Permite alinear icono y texto */
            align-items: center;
            justify-content: center;
        }
        .btn-action:hover {
            transform: translateY(-2px); /* Efecto de ligera elevación */
            box-shadow: 0 8px 15px rgba(0,0,0,0.2); /* Sombra más intensa */
        }

        /* Estilos para el botón "Añadir a Cotización" */
        .btn-add-quote { 
            background-color: var(--brand-dark-blue); 
            color: white; 
            border: 2px solid var(--brand-dark-blue);
        }
        .btn-add-quote:hover { 
            background-color: var(--brand-red-accent); 
            border-color: var(--brand-red-accent);
        }
        /* Estilo cuando el producto ya ha sido añadido */
        .btn-add-quote.added { 
            background-color: var(--brand-yellow); 
            color: var(--brand-dark-blue); 
            cursor: not-allowed; /* Cambia el cursor para indicar que no es clickeable */
            border-color: var(--brand-yellow);
        }

        /* Estilos para el botón "Mi Cotización" / "Enviar Solicitud" */
        .btn-submit-quote { 
            background-color: var(--brand-yellow); 
            color: var(--brand-dark-blue); 
            border: 2px solid var(--brand-yellow);
        }
        .btn-submit-quote:hover { 
            background-color: var(--brand-dark-blue); 
            color: var(--brand-yellow); 
            border-color: var(--brand-dark-blue);
        }

        /* Estilos para el menú desplegable de categoría */
        .category-dropdown {
            padding: 0.6rem 1.2rem;
            border: 2px solid var(--brand-dark-blue);
            color: var(--brand-dark-blue);
            background-color: white; /* Fondo blanco para el dropdown */
            border-radius: 0.5rem; /* Bordes redondeados */
            font-weight: 700;
            transition: all 0.2s ease-in-out;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            appearance: none; /* Elimina estilos nativos del select */
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 20 20' fill='currentColor'%3E%3Cpath fill-rule='evenodd' d='M5.293 7.293a1 1 0 011.414 0L10 10.586l3.293-3.293a1 1 0 111.414 1.414l-4 4a1 1 0 01-1.414 0l-4-4a1 1 0 010-1.414z' clip-rule='evenodd'/%3E%3C/svg%3E"); /* Icono de flecha */
            background-repeat: no-repeat;
            background-position: right 0.75rem center;
            background-size: 1.5em 1.5em;
            padding-right: 2.5rem; /* Espacio para el icono */
        }
        .category-dropdown:hover { 
            border-color: var(--brand-red-accent); 
            box-shadow: 0 4px 8px rgba(0,0,0,0.15);
        }
        .category-dropdown:focus {
            outline: none;
            border-color: var(--brand-yellow);
            box-shadow: 0 0 0 3px rgba(255, 200, 87, 0.3);
        }

        /* Estilos para el contenedor del slider de imágenes del producto */
        .product-image-slider { 
            position: relative; 
            overflow: hidden; 
            border-top-left-radius: 0.75rem; /* Coincide con el radio de la tarjeta */
            border-top-right-radius: 0.75rem; /* Coincide con el radio de la tarjeta */
            height: 18rem; /* Altura fija para consistencia visual */
            display: flex; /* Asegura que las imágenes estén alineadas */
            align-items: center; /* Centra las imágenes verticalmente */
            justify-content: center; /* Centra las imágenes horizontalmente */
            touch-action: pan-y; /* Permite el desplazamiento vertical de la página mientras se desliza horizontalmente */
        }
        .slider-wrapper { 
            display: flex; 
            transition: transform 0.3s ease-in-out; /* Transición para el movimiento del slider */
            width: 100%; 
            height: 100%; 
        }
        .slider-wrapper img { 
            width: 100%; 
            height: 100%; 
            object-fit: cover; /* Cubre el área manteniendo el aspecto */
            flex-shrink: 0; /* Evita que las imágenes se encojan */
        }
        /* Estilos para los botones de navegación del slider (flechas) */
        .slider-nav {
            position: absolute; top: 50%; transform: translateY(-50%);
            background-color: rgba(7, 22, 45, 0.6); /* Fondo semi-transparente */
            color: white;
            border-radius: 50%; /* Forma circular */
            width: 2.8rem; height: 2.8rem; 
            display: flex; align-items: center; justify-content: center;
            cursor: pointer;
            transition: background-color 0.2s, transform 0.2s;
            font-size: 1.2rem;
            z-index: 10; /* Asegura que los botones estén sobre las imágenes */
        }
        .slider-nav:hover { 
            background-color: rgba(7, 22, 45, 0.9); /* Más opaco al pasar el mouse */
            transform: translateY(-50%) scale(1.1); /* Efecto de escala */
        }
        .slider-nav.prev { left: 0.75rem; } /* Posición del botón anterior */
        .slider-nav.next { right: 0.75rem; } /* Posición del botón siguiente */

        /* Estilos para los puntos de navegación del slider */
        .slider-dots {
            position: absolute; bottom: 0.75rem; left: 50%;
            transform: translateX(-50%);
            display: flex; gap: 0.6rem; /* Espaciado entre los puntos */
            z-index: 10; /* Asegura que los puntos estén sobre las imágenes */
        }
        .slider-dots .dot {
            width: 0.85rem; height: 0.85rem;
            background-color: rgba(255, 255, 255, 0.6); /* Puntos semi-transparentes */
            border-radius: 50%; /* Forma circular */
            transition: background-color 0.2s, transform 0.2s;
            cursor: pointer;
            border: 1px solid rgba(0,0,0,0.1);
        }
        .slider-dots .dot.active { 
            background-color: var(--brand-yellow); /* Color activo */
            transform: scale(1.2); /* Efecto de escala para el punto activo */
            border-color: var(--brand-dark-blue);
        }

        /* Estilos para el fondo oscuro del panel lateral de cotización */
        .quote-panel-backdrop {
            position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background-color: rgba(0, 0, 0, 0.7); /* Fondo oscuro semi-transparente */
            display: flex; 
            justify-content: flex-end; /* Alinea el panel a la derecha */
            align-items: flex-start; /* Alinea el panel en la parte superior */
            z-index: 50; /* Asegura que esté sobre el contenido principal */
            opacity: 0; /* Inicialmente invisible */
            visibility: hidden; /* Inicialmente oculto */
            transition: opacity 0.3s ease, visibility 0.3s ease; /* Transición para aparecer/desaparecer */
        }
        .quote-panel-backdrop.show {
            opacity: 1; /* Visible */
            visibility: visible; /* Muestra el elemento */
        }
        /* Estilos para el contenido del panel lateral de cotización */
        .quote-panel-content {
            background-color: white;
            border-radius: 0; /* Eliminado el radio */
            box-shadow: 0 25px 50px -12px rgb(0 0 0 / 0.35); /* Sombra */
            text-align: left;
            width: 100%; /* Ancho completo en pantallas muy pequeñas */
            max-width: 600px; /* Ancho máximo para pantallas más grandes, aumentado */
            height: 100vh; /* Ocupa toda la altura de la ventana */
            overflow-y: auto; /* Permite desplazamiento vertical si el contenido es largo */
            transform: translateX(100%); /* Inicialmente fuera de la pantalla a la derecha */
            transition: transform 0.3s ease-in-out; /* Transición para el deslizamiento */
            padding: 1.5rem; 
        }
        /* Ajustes de ancho responsivo para el panel lateral */
        @media (min-width: 640px) { /* Para pantallas pequeñas (sm) y superiores */
            .quote-panel-content {
                width: 80%; /* Ocupa el 80% del ancho */
            }
        }
        @media (min-width: 768px) { /* Para pantallas medianas (md) y superiores */
            .quote-panel-content {
                width: 600px; /* Vuelve al ancho fijo máximo */
            }
        }
        /* Estado 'show' para el contenido del panel, deslizándolo a la vista */
        .quote-panel-backdrop.show .quote-panel-content {
            transform: translateX(0); /* Desliza el panel a la posición visible */
        }

        /* Estilos para campos de formulario (input y textarea) */
        input[type="text"], input[type="email"], input[type="tel"], input[type="date"], textarea {
            border-width: 2px;
            border-color: #e2e8f0; /* Color de borde por defecto */
            border-radius: 0.5rem;
            padding: 0.75rem 1rem; 
            transition: all 0.2s ease-in-out; /* Transición para el foco */
        }
        input[type="text"]:focus, input[type="email"]:focus, input[type="tel"]:focus, input[type="date"]:focus, textarea:focus {
            outline: none; /* Elimina el contorno de foco por defecto */
            border-color: var(--brand-yellow); /* Color de borde al enfocar */
            box-shadow: 0 0 0 3px rgba(255, 200, 87, 0.3); /* Sombra suave al enfocar */
        }

        /* Estilos de la barra de desplazamiento para el resumen de la cotización */
        #quote-summary::-webkit-scrollbar {
            width: 8px;
        }
        #quote-summary::-webkit-scrollbar-track {
            background: var(--brand-light-gray);
            border-radius: 10px;
        }
        #quote-summary::-webkit-scrollbar-thumb {
            background: var(--brand-dark-blue);
            border-radius: 10px;
        }
        #quote-summary::-webkit-scrollbar-thumb:hover {
            background: var(--brand-red-accent);
        }

        /* Estilos para el fondo oscuro de la caja de mensajes personalizada */
        .message-box-backdrop {
            position: fixed; top: 0; left: 0; width: 100%; height: 100%;
            background-color: rgba(0, 0, 0, 0.7);
            display: flex; justify-content: center; align-items: center; z-index: 100; /* Alto z-index para estar sobre todo */
            opacity: 0;
            visibility: hidden;
            transition: opacity 0.3s ease, visibility 0.3s ease;
        }
        .message-box-backdrop.show {
            opacity: 1;
            visibility: visible;
        }
        /* Estilos para el contenido de la caja de mensajes */
        .message-box-content {
            background-color: white; padding: 2rem; border-radius: 0.75rem;
            box-shadow: 0 25px 50px -12px rgb(0 0 0 / 0.35);
            text-align: center; max-width: 400px; width: 90%;
            transform: scale(0.9); /* Ligeramente más pequeño al inicio */
            transition: transform 0.3s ease, opacity 0.3s ease;
        }
        .message-box-backdrop.show .message-box-content {
            transform: scale(1); /* Escala normal al aparecer */
            opacity: 1;
        }

        /* Estilos para el botón "Volver Arriba" */
        #back-to-top-btn {
            position: fixed;
            bottom: 80px; /* Posición sobre el botón de cotización */
            right: 20px;
            background-color: var(--brand-red-accent);
            color: white;
            padding: 0.8rem 1rem;
            border-radius: 50%; /* Circular */
            font-size: 1.5rem;
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
            transition: opacity 0.3s ease, transform 0.3s ease;
            opacity: 0; /* Inicialmente oculto */
            visibility: hidden;
            z-index: 40; /* Sobre el contenido, bajo el panel de cotización */
            display: flex;
            align-items: center;
            justify-content: center;
        }
        #back-to-top-btn.show {
            opacity: 1; /* Visible */
            visibility: visible;
            transform: translateY(0);
        }
        #back-to-top-btn:hover {
            transform: translateY(-3px); /* Efecto de elevación */
            box-shadow: 0 6px 12px rgba(0,0,0,0.3);
        }

        /* Animación de pulso para el número de teléfono */
        @keyframes pulse-glow {
            0% {
                box-shadow: 0 0 0 0 rgba(255, 200, 87, 0.7); /* brand-yellow with opacity */
            }
            70% {
                box-shadow: 0 0 0 15px rgba(255, 200, 87, 0);
            }
            100% {
                box-shadow: 0 0 0 0 rgba(255, 200, 87, 0);
            }
        }

        /* Aplicar la animación al enlace del teléfono */
        .header-phone-link {
            animation: pulse-glow 2s infinite;
        }
    </style>
</head>
<body>

    <!-- Encabezado de la página con logotipo y número de teléfono -->
    <!-- El div exterior maneja el fondo de ancho completo, el div interior centra el contenido -->
    <div class="bg-white shadow-md relative z-50">
        <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-4 flex flex-col sm:flex-row sm:items-center sm:justify-between max-w-6xl">
            <div class="flex items-center flex-shrink-0 mb-2 sm:mb-0">
                <!-- SVG de logotipo mejorado con colores de la marca -->
                <svg class="h-12 w-auto" viewBox="0 0 48 48" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <circle cx="24" cy="24" r="20" fill="var(--brand-yellow)" stroke="var(--brand-dark-blue)" stroke-width="2"/>
                    <path d="M24 10C18.4772 10 14 14.4772 14 20C14 25.5228 18.4772 30 24 30C29.5228 30 34 25.5228 34 20C34 14.4772 29.5228 10 24 10Z" fill="var(--brand-dark-blue)"/>
                    <path d="M24 30V38" stroke="var(--brand-dark-blue)" stroke-width="4" stroke-linecap="round" stroke-linejoin="round"/>
                    <path d="M18 24L24 30L30 24" stroke="var(--brand-yellow)" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
                </svg>
                <!-- Nombre del negocio con estilos de marca -->
                <span class="text-3xl font-bold text-brand-dark-blue ml-3 font-heading">Tu Negocio</span>
            </div>
            <!-- Sección para el número de teléfono, ahora un enlace directo para llamar y más llamativo -->
            <a href="tel:+15551234567" class="header-phone-link flex items-center bg-brand-dark-blue text-brand-yellow px-5 py-3 rounded-full shadow-xl ml-0 sm:ml-4 flex-shrink-0 hover:scale-105 hover:shadow-2xl transform transition-all duration-300 ease-in-out no-underline">
                <i class="fa-solid fa-phone mr-3 text-3xl"></i>
                <span class="text-2xl md:text-3xl font-extrabold">+1 (555) 123-4567</span>
            </a>
        </div>
    </div>

    <!-- Espacio para imagen entre el encabezado y el bloque de bienvenida - Ahora de lado a lado -->
    <div class="w-full">
        <img src="https://placehold.co/1200x150/ffc857/07162d?text=ESPACIO+PUBLICITARIO" alt="Espacio publicitario" class="w-full h-auto object-cover">
    </div>

    <!-- Sección de Bienvenida - Fondo azul oscuro de lado a lado y sin bordes redondeados -->
    <section class="bg-[color:var(--brand-dark-blue)] py-4 sm:py-3 text-center relative z-20">
        <h1 class="text-3xl sm:text-4xl font-heading mb-0 leading-tight text-white">
            ¡Bienvenido, <span class="text-[var(--brand-yellow)]">Nombre del Cliente</span>!
        </h1>
        <p class="text-md sm:text-lg max-w-2xl mx-auto text-white mt-1">
            Explora nuestro catálogo para tu evento perfecto.
        </p>
    </section>

    <!-- Contenedor principal de la aplicación - El div exterior es de ancho completo, el interior centra el contenido -->
    <div class="w-full">
        <div id="app" class="container mx-auto px-4 sm:px-6 lg:px-8 py-4 max-w-7xl">

            <!-- Encabezado de la sección de productos -->
            <header class="text-center mb-8 mt-8" id="products-section">
                <h2 class="text-3xl md:text-5xl font-heading text-brand-dark-blue">Nuestros Productos</h2>
                <p class="text-gray-600 text-lg mt-2">Selecciona los productos que te interesan para crear una cotización personalizada.</p>
            </header>
            
            <!-- Contenedor para el menú desplegable de filtro de categoría -->
            <div class="flex justify-center mb-10">
                <select id="category-dropdown" class="category-dropdown w-full max-w-xs md:max-w-sm">
                    <!-- Las opciones se generarán dinámicamente con JavaScript -->
                </select>
            </div>

            <!-- Vista principal del catálogo de productos -->
            <main id="catalog-view" class="fade-in">
                <!-- Cuadrícula donde se renderizarán los productos -->
                <div id="product-grid" class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8"></div>
            </main>

            <!-- Botón flotante "Mi Cotización" - Inicialmente oculto con la clase 'hidden' -->
            <button id="view-quote-btn" class="fixed bottom-5 right-5 btn-action btn-submit-quote flex items-center justify-center z-40 hidden">
                <i class="fa-solid fa-clipboard-list mr-2"></i>
                <span>Mi Cotización</span>
                <span id="quote-count" class="ml-2 bg-brand-red-accent text-white text-xs font-bold rounded-full h-6 w-6 flex items-center justify-center">0</span>
            </button>

            <!-- Botón flotante "Volver Arriba" - Inicialmente oculto -->
            <button id="back-to-top-btn" class="hidden">
                <i class="fa-solid fa-arrow-up"></i>
            </button>

            <!-- Panel lateral del formulario de cotización -->
            <div id="quote-form-view" class="quote-panel-backdrop">
                <div id="modal-content-wrapper" class="quote-panel-content">
                    <div class="flex justify-between items-start mb-6">
                        <div>
                            <h2 class="text-2xl font-heading text-brand-dark-blue">Solicitar Cotización</h2>
                            <p class="text-gray-500">Completa tus datos para enviarte la información de los productos seleccionados.</p>
                        </div>
                        <!-- Botón para cerrar el panel lateral -->
                        <button id="back-to-catalog-btn" class="text-gray-500 hover:text-brand-dark-blue text-3xl font-light leading-none">&times;</button>
                    </div>
                    <!-- Resumen de productos seleccionados en la cotización -->
                    <div id="quote-summary" class="mb-8 max-h-64 overflow-y-auto pr-2">
                        <h3 class="text-lg font-heading border-b pb-2 mb-4 text-brand-dark-blue">Productos Seleccionados:</h3>
                        <div id="selected-products-list" class="space-y-4"></div>
                        <p id="empty-quote-message" class="text-center text-gray-500 py-4 hidden">Aún no has seleccionado productos.</p>
                    </div>
                    <!-- Formulario de contacto para la cotización -->
                    <form id="contact-form">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <input type="text" id="name" name="name" required placeholder="Nombre Completo" class="w-full">
                            <input type="email" id="email" name="email" required placeholder="Correo Electrónico" class="w-full">
                            <input type="tel" id="phone" name="phone" placeholder="Teléfono" class="w-full" pattern="[0-9]{8,15}" title="Por favor, introduce un número de teléfono válido (8-15 dígitos).">
                            <input type="date" id="event-date" name="event-date" required class="w-full">
                            <div class="md:col-span-2">
                                <textarea id="comments" name="comments" rows="4" placeholder="Comentarios Adicionales (ej. fecha del evento, cantidad, etc.)" class="w-full"></textarea>
                            </div>
                        </div>
                        <div class="mt-8 text-right">
                            <!-- Botón para enviar la solicitud de cotización -->
                            <button type="submit" id="submit-quote-btn" class="btn-action btn-submit-quote">
                                <i class="fa-solid fa-paper-plane mr-2" id="submit-icon"></i>
                                <span id="submit-text">Enviar Solicitud</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Caja de Mensajes Personalizada (reemplaza a alert()) -->
            <div id="message-box-backdrop" class="message-box-backdrop">
                <div id="message-box-content" class="message-box-content">
                    <h3 id="message-box-title" class="text-xl font-heading text-brand-dark-blue mb-4"></h3>
                    <p id="message-box-text" class="text-gray-700 mb-6"></p>
                    <button id="message-box-ok-btn" class="btn-action btn-add-quote">Aceptar</button>
                </div>
            </div>

        </div>
    </div>

    <!-- Pie de página -->
    <footer class="mt-16 py-8" style="background-color: var(--brand-darkest-blue);">
        <div class="container mx-auto px-4 sm:px-6 lg:px-8 text-center text-gray-400">
            <p>&copy; 2024 Nombre de tu Negocio. Todos los derechos reservados.</p>
            <p class="text-sm mt-2">Hecho con <i class="fa-solid fa-heart text-red-500"></i> para eventos increíbles.</p>
        </div>
    </footer>

    <script>
        document.addEventListener('DOMContentLoaded', () => {
            // --- DATOS DE EJEMPLO DE PRODUCTOS ---
            // Array que contiene la información detallada de cada producto.
            // Cada objeto de producto incluye su ID, nombre, descripción, categoría, código
            // y un array de URLs de imágenes para el slider.
            const products = [
                { id: 1, name: 'Silla Tiffany Blanca', description: 'Silla elegante y clásica, perfecta para bodas y eventos formal es.', category: 'Sillas', code: 'SITIF001', imageUrls: ['https://placehold.co/600x600/FFFFFF/333333?text=Silla+Tiffany+1', 'https://placehold.co/600x600/FAFAFA/333333?text=Silla+Tiffany+2', 'https://placehold.co/600x600/F0F0F0/333333?text=Silla+Tiffany+3'] },
                { id: 2, name: 'Mesa Redonda Imperial', description: 'Mesa de gran tamaño ideal para banquetes y reuniones, con capacidad para 10-12 personas.', category: 'Mesas', code: 'MESAIMP002', imageUrls: ['https://placehold.co/600x600/E0E0E0/333333?text=Mesa+Redonda+1', 'https://placehold.co/600x600/DADADA/333333?text=Mesa+Redonda+2'] },
                { id: 3, name: 'Carpa Elegante 10x5m', description: 'Carpa resistente y espaciosa, ideal para eventos al aire libre, protege del sol y la lluvia.', category: 'Carpas', code: 'CARPAELG003', imageUrls: ['https://placehold.co/600x600/F0F0F0/333333?text=Carpa+Elegante+1', 'https://placehold.co/600x600/E8E8E8/333333?text=Carpa+Elegante+2', 'https://placehold.co/600x600/F5F5F5/333333?text=Carpa+Elegante+3'] },
                { id: 4, name: 'Mantelería de Lino', description: 'Manteles de lino de alta calidad, disponibles en varios colores para complementar tu decoración.', category: 'Accesorios', code: 'MANTLINO004', imageUrls: ['https://placehold.co/600x600/DDDDDD/333333?text=Manteleria+Lino'] },
                { id: 5, name: 'Silla Avant Garde', description: 'Silla moderna y cómoda, perfecta para eventos contemporáneos y conferencias.', category: 'Sillas', code: 'SIAVG005', imageUrls: ['https://placehold.co/600x600/FFFFFF/333333?text=Silla+Avant+Garde+A', 'https://placehold.co/600x600/FAFAFA/333333?text=Silla+Avant+Garde+B'] },
                { id: 6, name: 'Mesa Rectangular de Madera', description: 'Mesa robusta de madera, ideal para cenas rústicas o estaciones de comida.', category: 'Mesas', code: 'MESAMAD006', imageUrls: ['https://placehold.co/600x600/D2B48C/FFFFFF?text=Mesa+Madera+1', 'https://placehold.co/600x600/C1A37E/FFFFFF?text=Mesa+Madera+2'] },
                { id: 7, name: 'Pista de Baile Iluminada', description: 'Pista de baile con luces LED integradas, crea un ambiente vibrante y divertido.', category: 'Pisos y Escenarios', code: 'PISTAB007', imageUrls: ['https://placehold.co/600x600/333333/FFFFFF?text=Pista+Baile+1', 'https://placehold.co/600x600/444444/FFFFFF?text=Pista+Baile+2', 'https://placehold.co/600x600/555555/FFFFFF?text=Pista+Baile+3'] },
                { id: 8, name: 'Centros de Mesa Florales', description: 'Arreglos florales personalizados para embellecer tus mesas y espacios.', category: 'Decoración', code: 'CMFLOR008', imageUrls: ['https://placehold.co/600x600/90EE90/333333?text=Flores+1'] },
                { id: 9, name: 'Iluminación Ambiental LED', description: 'Sistema de iluminación LED para crear ambientes únicos y destacar elementos decorativos.', category: 'Decoración', code: 'ILLUMLED009', imageUrls: ['https://placehold.co/600x600/ADD8E6/333333?text=Iluminacion+LED'] },
                { id: 10, name: 'Vajilla Premium', description: 'Juego de vajilla de porcelana fina, perfecto para cenas de gala y eventos exclusivos.', category: 'Accesorios', code: 'VAJILLAP010', imageUrls: ['https://placehold.co/600x600/F8F8F8/333333?text=Vajilla+1'] },
            ];

            // Array para almacenar los productos que el usuario ha añadido a su cotización.
            let quoteItems = [];
            // Variable para mantener el filtro de categoría actual ('Todos' por defecto).
            let currentFilter = 'Todos';

            // --- REFERENCIAS A ELEMENTOS DEL DOM ---
            // Se obtienen referencias a los elementos HTML que serán manipulados por JavaScript.
            const productGrid = document.getElementById('product-grid');
            const quoteCount = document.getElementById('quote-count');
            const quoteFormView = document.getElementById('quote-form-view');
            const viewQuoteBtn = document.getElementById('view-quote-btn');
            const backToCatalogBtn = document.getElementById('back-to-catalog-btn');
            const selectedProductsList = document.getElementById('selected-products-list');
            const emptyQuoteMessage = document.getElementById('empty-quote-message');
            const contactForm = document.getElementById('contact-form');
            const submitQuoteBtn = document.getElementById('submit-quote-btn');
            const submitIcon = document.getElementById('submit-icon');
            const submitText = document.getElementById('submit-text');
            // Se actualiza la referencia para el nuevo dropdown
            const categoryDropdown = document.getElementById('category-dropdown'); 
            const backToTopBtn = document.getElementById('back-to-top-btn');

            // Elementos para la caja de mensajes personalizada (reemplazo de alert()).
            const messageBoxBackdrop = document.getElementById('message-box-backdrop');
            const messageBoxTitle = document.getElementById('message-box-title');
            const messageBoxText = document.getElementById('message-box-text');
            const messageBoxOkBtn = document.getElementById('message-box-ok-btn');

            /**
             * Muestra una caja de mensajes personalizada en lugar de la alerta del navegador.
             * Esto mejora la experiencia de usuario y el control sobre el estilo.
             * @param {string} title - El título de la caja de mensajes.
             * @param {string} message - El contenido del mensaje.
             */
            function showMessageBox(title, message) {
                messageBoxTitle.textContent = title;
                messageBoxText.textContent = message;
                messageBoxBackdrop.classList.add('show');
            }

            // Event listener para el botón "Aceptar" en la caja de mensajes.
            // Al hacer clic, oculta la caja de mensajes.
            messageBoxOkBtn.addEventListener('click', () => {
                messageBoxBackdrop.classList.remove('show');
            });

            // --- LÓGICA DEL SLIDER DE IMÁGENES PARA CADA PRODUCTO ---
            /**
             * Configura la funcionalidad del slider de imágenes para un elemento de producto dado.
             * Esto incluye la navegación con flechas y los puntos indicadores, y ahora también el deslizamiento táctil.
             * @param {HTMLElement} sliderElement - El contenedor del slider de imágenes de un producto.
             */
            const setupSlider = (sliderElement) => {
                const wrapper = sliderElement.querySelector('.slider-wrapper');
                const dotsContainer = sliderElement.querySelector('.slider-dots');
                const prevBtn = sliderElement.querySelector('.slider-nav.prev');
                const nextBtn = sliderElement.querySelector('.slider-nav.next');
                const slides = wrapper.querySelectorAll('img');
                let currentIndex = 0;
                let startX = 0; // Para el inicio del toque
                let currentTranslate = 0; // Para el desplazamiento actual durante el toque
                let isDragging = false; // Bandera para saber si se está arrastrando

                // Función para actualizar la visualización del slider (posición y puntos activos).
                function updateSlider() {
                    currentTranslate = -currentIndex * 100;
                    wrapper.style.transform = `translateX(${currentTranslate}%)`;
                    // Actualiza la clase 'active' en los puntos de navegación.
                    if (dotsContainer) {
                        dotsContainer.querySelectorAll('.dot').forEach((dot, index) => {
                            dot.classList.toggle('active', index === currentIndex);
                        });
                    }
                }

                // Event listener para la navegación por puntos.
                if (dotsContainer) {
                    dotsContainer.addEventListener('click', (e) => {
                        if (e.target.classList.contains('dot')) {
                            currentIndex = parseInt(e.target.dataset.index);
                            updateSlider();
                        }
                    });
                }

                // Event listener para el botón de navegación "anterior".
                if (prevBtn) {
                    prevBtn.addEventListener('click', () => {
                        currentIndex = (currentIndex > 0) ? currentIndex - 1 : slides.length - 1;
                        updateSlider();
                    });
                }

                // Event listener para el botón de navegación "siguiente".
                if (nextBtn) {
                    nextBtn.addEventListener('click', () => {
                        currentIndex = (currentIndex < slides.length - 1) ? currentIndex + 1 : 0;
                        updateSlider();
                    });
                }

                // --- Lógica para el deslizamiento (swipe) táctil ---
                sliderElement.addEventListener('touchstart', (e) => {
                    startX = e.touches[0].clientX; // Guarda la posición X inicial del toque
                    isDragging = true;
                    wrapper.style.transition = 'none'; // Deshabilita la transición CSS durante el arrastre
                });

                sliderElement.addEventListener('touchmove', (e) => {
                    if (!isDragging) return;
                    const currentX = e.touches[0].clientX;
                    const diffX = currentX - startX; // Diferencia de desplazamiento
                    // Calcula la nueva posición del wrapper, permitiendo un arrastre visual
                    wrapper.style.transform = `translateX(calc(${currentTranslate}% + ${diffX}px))`;
                });

                sliderElement.addEventListener('touchend', (e) => {
                    if (!isDragging) return;
                    isDragging = false;
                    wrapper.style.transition = 'transform 0.3s ease-in-out'; // Vuelve a habilitar la transición

                    const endX = e.changedTouches[0].clientX;
                    const swipeDistance = endX - startX; // Distancia total del deslizamiento
                    const threshold = 50; // Umbral en píxeles para considerar un deslizamiento

                    if (swipeDistance > threshold) {
                        // Deslizamiento hacia la derecha (anterior)
                        currentIndex = (currentIndex > 0) ? currentIndex - 1 : slides.length - 1;
                    } else if (swipeDistance < -threshold) {
                        // Deslizamiento hacia la izquierda (siguiente)
                        currentIndex = (currentIndex < slides.length - 1) ? currentIndex + 1 : 0;
                    }
                    updateSlider(); // Actualiza el slider a la nueva posición (o la misma si no hubo deslizamiento suficiente)
                });
            };

            /**
             * Renderiza la cuadrícula de productos basándose en el filtro de categoría activo.
             * Crea las tarjetas de producto dinámicamente y las inserta en el DOM.
             * @param {string} filter - La categoría a filtrar, o 'Todos' para mostrar todos los productos.
             */
            function renderProducts(filter = 'Todos') {
                productGrid.innerHTML = ''; // Limpia los productos existentes en la cuadrícula.
                // Filtra los productos según la categoría seleccionada.
                const filteredProducts = filter === 'Todos' ? products : products.filter(p => p.category === filter);

                // Muestra un mensaje si no hay productos en la categoría seleccionada.
                if (filteredProducts.length === 0) {
                    productGrid.innerHTML = '<p class="text-center text-gray-500 col-span-full py-10">No hay productos en esta categoría.</p>';
                    return;
                }

                // Itera sobre los productos filtrados para crear y añadir sus tarjetas al DOM.
                filteredProducts.forEach(product => {
                    // Verifica si el producto ya está en la cotización para actualizar el botón.
                    const isAdded = quoteItems.some(item => item.id === product.id);
                    const buttonText = isAdded ? 'Añadido' : 'Añadir a Cotización';
                    const buttonClasses = isAdded ? 'added' : '';

                    // Genera el HTML para las imágenes del slider y los puntos indicadores.
                    // Se incluye un manejador de errores para las imágenes.
                    const imagesHtml = product.imageUrls.map(url => `<img src="${url}" alt="${product.name}" class="w-full h-full object-cover" onerror="this.onerror=null;this.src='https://placehold.co/600x600/CCCCCC/666666?text=Imagen+No+Disponible';">`).join('');
                    const dotsHtml = product.imageUrls.length > 1 ? product.imageUrls.map((_, index) => `<div class="dot ${index === 0 ? 'active' : ''}" data-index="${index}"></div>`).join('') : '';

                    const productCard = document.createElement('div');
                    productCard.className = 'product-card fade-in'; // Añade animación de aparición.
                    productCard.innerHTML = `
                        <div class="product-image-slider">
                            <div class="slider-wrapper">${imagesHtml}</div>
                            ${product.imageUrls.length > 1 ? `
                                <div class="slider-nav prev"><i class="fa-solid fa-chevron-left"></i></div>
                                <div class="slider-nav next"><i class="fa-solid fa-chevron-right"></i></div>
                                <div class="slider-dots">${dotsHtml}</div>
                            ` : ''}
                        </div>
                        <div class="p-4 flex flex-col flex-grow">
                            <h3 class="title mb-2">${product.name}</h3>
                            <p class="text-sm text-gray-700 mb-2">${product.description}</p>
                            <p class="text-sm font-semibold text-brand-dark-blue mb-1 flex items-center justify-center">
                                <i class="fa-solid fa-tag mr-2 text-brand-red-accent"></i>Categoría: ${product.category}
                            </p>
                            <p class="text-sm font-semibold text-brand-dark-blue mb-4 flex items-center justify-center">
                                <i class="fa-solid fa-barcode mr-2 text-brand-red-accent"></i>Código: ${product.code}
                            </p>
                            <button data-product-id="${product.id}" class="btn-action btn-add-quote w-full mt-auto ${buttonClasses}" ${isAdded ? 'disabled' : ''}>
                                ${buttonText}
                            </button>
                        </div>
                    `;
                    productGrid.appendChild(productCard);

                    // Inicializa el slider para esta tarjeta si tiene múltiples imágenes.
                    if (product.imageUrls.length > 1) {
                        setupSlider(productCard.querySelector('.product-image-slider'));
                    }
                });
            }

            /**
             * Renderiza las opciones del menú desplegable de categoría.
             * Genera una opción para cada categoría única de productos, más una opción "Todos".
             */
            function renderCategoryDropdown() {
                // Obtiene las categorías únicas de los productos y añade 'Todos'.
                const categories = ['Todos', ...new Set(products.map(p => p.category))];
                categoryDropdown.innerHTML = ''; // Limpia las opciones existentes.
                categories.forEach(category => {
                    const option = document.createElement('option');
                    option.value = category;
                    option.textContent = category;
                    // Marca la opción como seleccionada si coincide con el filtro actual.
                    if (category === currentFilter) {
                        option.selected = true;
                    }
                    categoryDropdown.appendChild(option);
                });
            }

            /**
             * Actualiza el contador de productos en la cotización y la visibilidad del botón "Mi Cotización".
             */
            function updateQuoteCount() {
                quoteCount.textContent = quoteItems.length;
                // Muestra u oculta el botón "Mi Cotización" según si hay productos en la lista.
                if (quoteItems.length > 0) {
                    viewQuoteBtn.classList.remove('hidden');
                } else {
                    viewQuoteBtn.classList.add('hidden'); 
                }
            }

            /**
             * Muestra el panel lateral del formulario de solicitud de cotización.
             * También renderiza el resumen de la cotización dentro del panel.
             */
            function showQuoteForm() {
                quoteFormView.classList.add('show');
                renderQuoteSummary(); // Renderiza el resumen de la cotización al abrir el panel.
            }

            /**
             * Oculta el panel lateral del formulario de solicitud de cotización.
             */
            function hideQuoteForm() {
                quoteFormView.classList.remove('show');
            }
            
            /**
             * Renderiza la lista de productos seleccionados en el resumen de la cotización dentro del panel lateral.
             * Muestra un mensaje si la cotización está vacía.
             */
            function renderQuoteSummary() {
                selectedProductsList.innerHTML = ''; // Limpia la lista existente.
                // Si no hay productos en la cotización, muestra un mensaje y deshabilita el botón de enviar.
                if (quoteItems.length === 0) {
                    emptyQuoteMessage.classList.remove('hidden');
                    submitQuoteBtn.disabled = true;
                } else {
                    emptyQuoteMessage.classList.add('hidden');
                    submitQuoteBtn.disabled = false;
                    // Itera sobre los productos en la cotización para mostrarlos en el resumen.
                    quoteItems.forEach(item => {
                        const listItem = document.createElement('div');
                        listItem.className = 'flex items-center justify-between bg-gray-50 p-3 rounded-md shadow-sm';
                        listItem.innerHTML = `
                            <div class="flex items-center">
                                <img src="${item.imageUrls[0]}" alt="${item.name}" class="w-16 h-16 object-cover rounded-md mr-4" onerror="this.onerror=null;this.src='https://placehold.co/600x600/CCCCCC/666666?text=Imagen+No+Disponible';">
                                <div>
                                    <p class="font-semibold text-brand-dark-blue">${item.name}</p>
                                    <p class="text-sm text-gray-500 flex items-center"><i class="fa-solid fa-tag mr-2 text-brand-red-accent"></i>Categoría: ${item.category}</p>
                                    <p class="text-sm text-gray-500 flex items-center"><i class="fa-solid fa-barcode mr-2 text-brand-red-accent"></i>Código: ${item.code}</p>
                                </div>
                            </div>
                            <button data-remove-id="${item.id}" class="font-semibold text-sm hover:underline" style="color: var(--brand-red-accent);">Quitar</button>
                        `;
                        selectedProductsList.appendChild(listItem);
                    });
                }
            }
            
            // --- GESTIÓN DE EVENTOS ---

            // Listener para añadir productos a la cotización desde la cuadrícula de productos.
            // Utiliza delegación de eventos para manejar clics en los botones "Añadir a Cotización".
            productGrid.addEventListener('click', (e) => {
                const button = e.target.closest('.btn-add-quote');
                if (button && !button.disabled) {
                    const productId = parseInt(button.dataset.productId);
                    const product = products.find(p => p.id === productId);
                    if (product) {
                        quoteItems.push(product);
                        updateQuoteCount(); // Actualiza el contador de la cotización.
                        renderProducts(currentFilter); // Vuelve a renderizar para actualizar el estado del botón.
                    }
                }
            });

            // Listener para quitar productos del resumen de la cotización en el panel lateral.
            // Utiliza delegación de eventos para manejar clics en los botones "Quitar".
            selectedProductsList.addEventListener('click', (e) => {
                if(e.target.dataset.removeId) {
                    const productId = parseInt(e.target.dataset.removeId);
                    // Filtra el array para eliminar el producto con el ID correspondiente.
                    quoteItems = quoteItems.filter(item => item.id !== productId);
                    renderQuoteSummary(); // Vuelve a renderizar el resumen.
                    updateQuoteCount(); // Actualiza el contador.
                    renderProducts(currentFilter); // Vuelve a renderizar los productos para restablecer los botones "Añadido".
                }
            });

            // Listener para el menú desplegable de categoría.
            // Actualiza el filtro actual y renderiza los productos.
            categoryDropdown.addEventListener('change', (e) => {
                currentFilter = e.target.value; // Establece el filtro actual desde el valor seleccionado.
                renderProducts(currentFilter); // Renderiza los productos con el nuevo filtro.
            });

            // Listeners para mostrar y ocultar el formulario de cotización.
            viewQuoteBtn.addEventListener('click', showQuoteForm);
            backToCatalogBtn.addEventListener('click', hideQuoteForm);
            // Cierra el panel si se hace clic en el fondo oscuro (fuera del contenido del panel).
            quoteFormView.addEventListener('click', (e) => {
                if (e.target === quoteFormView) {
                    hideQuoteForm();
                }
            });

            // Lógica para mostrar/ocultar el botón "Volver Arriba" al hacer scroll.
            window.addEventListener('scroll', () => {
                if (window.scrollY > 300) { // Muestra el botón después de desplazar 300px.
                    backToTopBtn.classList.add('show');
                } else {
                    backToTopBtn.classList.remove('show');
                }
            });

            // Listener para el botón "Volver Arriba" para desplazar la página al inicio.
            backToTopBtn.addEventListener('click', () => {
                window.scrollTo({ top: 0, behavior: 'smooth' }); // Desplazamiento suave.
            });

            // Event listener para el envío del formulario de contacto.
            // Realiza validaciones y simula el envío de datos.
            contactForm.addEventListener('submit', async (e) => {
                e.preventDefault(); // Previene el envío por defecto del formulario.

                // Validación básica de campos obligatorios.
                const name = document.getElementById('name').value.trim();
                const email = document.getElementById('email').value.trim();
                const eventDate = document.getElementById('event-date').value.trim();

                if (!name || !email || !eventDate) {
                    showMessageBox('Error de Formulario', 'Por favor, completa todos los campos obligatorios (Nombre, Correo Electrónico, Fecha del Evento).');
                    return;
                }

                // Validación simple del formato de correo electrónico usando una expresión regular.
                const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                if (!emailPattern.test(email)) {
                    showMessageBox('Error de Formulario', 'Por favor, introduce un formato de correo electrónico válido.');
                    return;
                }

                // Verifica si hay productos en la cotización antes de permitir el envío.
                if (quoteItems.length === 0) {
                    showMessageBox('Error de Cotización', 'Por favor, añade al menos un producto a tu cotización antes de enviarla.');
                    return;
                }

                // Muestra el estado de carga en el botón de envío.
                submitQuoteBtn.disabled = true;
                submitText.textContent = 'Enviando...';
                submitIcon.classList.remove('fa-paper-plane');
                submitIcon.classList.add('fa-spinner', 'fa-spin');

                // Simula una llamada a la API con un retardo para mostrar el estado de carga.
                await new Promise(resolve => setTimeout(resolve, 2000)); 

                // Muestra los datos de la solicitud en la consola (en un entorno real, estos datos
                // se enviarían a un servidor a través de una petición fetch/XMLHttpRequest).
                console.log('--- NUEVA SOLICITUD DE COTIZACIÓN ---');
                console.log('Datos del Cliente:', {
                    name: document.getElementById('name').value,
                    email: document.getElementById('email').value,
                    phone: document.getElementById('phone').value,
                    eventDate: document.getElementById('event-date').value,
                    comments: document.getElementById('comments').value
                });
                console.log('Productos Solicitados:', quoteItems);

                // Muestra un mensaje de éxito al usuario.
                showMessageBox('¡Solicitud Enviada!', 'Tu solicitud de cotización ha sido recibida con éxito. Nos pondremos en contacto contigo pronto.');
                
                hideQuoteForm(); // Oculta el panel del formulario.
                quoteItems = []; // Limpia la cotización después del envío exitoso.
                contactForm.reset(); // Reinicia los campos del formulario.
                updateQuoteCount(); // Actualiza el contador de la cotización a 0.
                renderProducts(currentFilter); // Vuelve a renderizar los productos para restablecer los botones "Añadido".

                // Restablece el estado del botón de envío a su estado original.
                submitQuoteBtn.disabled = false;
                submitText.textContent = 'Enviar Solicitud';
                submitIcon.classList.remove('fa-spinner', 'fa-spin');
                submitIcon.classList.add('fa-paper-plane');
            });

            // --- LLAMADAS DE RENDERIZADO INICIALES ---
            // Estas funciones se ejecutan cuando el DOM ha sido completamente cargado
            // para inicializar la interfaz de usuario.
            renderCategoryDropdown(); // Muestra las opciones del menú desplegable de categoría.
            renderProducts(currentFilter); // Muestra todos los productos inicialmente.
            updateQuoteCount(); // Inicializa el contador de la cotización (y oculta el botón si está en 0).
        });
    </script>
</body>
</html>
